// ===== STARBUCKS HACKER GAME - MAIN LOGIC =====

// Game State
const gameState = {
    isConnected: false,
    securityBypassed: false,
    isHidden: false,
    detectionRisk: 10,
    coffeeStolen: 0,
    moneyStolen: 0,
    recipesAcquired: [],
    recipesSold: [],
    commandHistory: [],
    historyIndex: -1,
    gamePhase: 'menu', // menu, playing, arrested, won
    npcMood: 'excited',
    scanCount: 0,
    wrongCommands: 0,
    discoveredServers: [],
    connectedServer: null,
    currentDay: 1,
    purchasedServerPacks: [],
    serversDestroyed: false
};

// Available Starbucks Servers (discovered after scan)
const starbucksServers = [
    { ip: '192.168.CAFE.1', name: 'SBUX-CORP-MAINFRAME', type: 'main', secure: true },
    { ip: '192.168.CAFE.42', name: 'SECRET-RECIPES-DB', type: 'recipes', secure: true },
    { ip: '192.168.CAFE.69', name: 'FINANCE-SERVER', type: 'money', secure: true },
    { ip: '192.168.CAFE.13', name: 'HONEYPOT-TRAP', type: 'trap', secure: false },
    { ip: '10.0.LATTE.1', name: 'BACKUP-SERVER', type: 'backup', secure: true }
];

// Black Market Server Packs (purchasable)
const serverPacks = [
    {
        id: 'asia_pack',
        name: 'Asia Pacific Pack',
        price: 25000,
        moneyRange: [15000, 50000],
        servers: [
            { ip: '172.16.TOKYO.1', name: 'SBUX-JAPAN-HQ', type: 'main', secure: true },
            { ip: '172.16.TOKYO.88', name: 'MATCHA-SECRETS-DB', type: 'recipes', secure: true },
            { ip: '172.16.SEOUL.42', name: 'KOREA-FINANCE', type: 'money', secure: true, moneyAmount: 50000 }
        ]
    },
    {
        id: 'europe_pack',
        name: 'European Pack',
        price: 35000,
        moneyRange: [25000, 75000],
        servers: [
            { ip: '10.EU.LONDON.1', name: 'SBUX-EU-MAINFRAME', type: 'main', secure: true },
            { ip: '10.EU.PARIS.33', name: 'CROISSANT-RECIPES', type: 'recipes', secure: true },
            { ip: '10.EU.SWISS.77', name: 'SWISS-BANK-VAULT', type: 'money', secure: true, moneyAmount: 100000 }
        ]
    },
    {
        id: 'latam_pack',
        name: 'Latin America Pack',
        price: 50000,
        moneyRange: [50000, 150000],
        servers: [
            { ip: '200.BRAZIL.SAO.1', name: 'SBUX-BRAZIL-HQ', type: 'main', secure: true },
            { ip: '200.BRAZIL.SAO.55', name: 'BRAZILIAN-BEAN-SECRETS', type: 'recipes', secure: true },
            { ip: '200.MEXICO.DF.99', name: 'MEXICO-TREASURY', type: 'money', secure: true, moneyAmount: 150000 },
            { ip: '200.COLOMBIA.BOG.13', name: 'CARTEL-HONEYPOT', type: 'trap', secure: false }
        ]
    },
    {
        id: 'middle_east_pack',
        name: 'Middle East Pack',
        price: 100000,
        moneyRange: [100000, 300000],
        servers: [
            { ip: '88.DUBAI.GOLD.1', name: 'SBUX-DUBAI-PALACE', type: 'main', secure: true },
            { ip: '88.DUBAI.GOLD.777', name: 'GOLD-LATTE-FORMULA', type: 'recipes', secure: true },
            { ip: '88.DUBAI.GOLD.999', name: 'SHEIKH-BILLIONS', type: 'money', secure: true, moneyAmount: 300000 },
            { ip: '88.QATAR.OIL.42', name: 'QATAR-RESERVES', type: 'money', secure: true, moneyAmount: 250000 }
        ]
    },
    {
        id: 'secret_pack',
        name: 'Dark Web Intel Pack',
        price: 200000,
        moneyRange: [200000, 500000],
        servers: [
            { ip: '666.DARK.WEB.1', name: 'CEO-PRIVATE-SERVER', type: 'main', secure: true },
            { ip: '666.DARK.WEB.99', name: 'UNRELEASED-2025-MENU', type: 'recipes', secure: true },
            { ip: '666.DARK.WEB.777', name: 'OFFSHORE-MILLIONS', type: 'money', secure: true, moneyAmount: 500000 },
            { ip: '666.DARK.WEB.13', name: 'FBI-HONEYPOT-ELITE', type: 'trap', secure: false }
        ]
    },
    {
        id: 'corporate_pack',
        name: 'Corporate HQ Pack',
        price: 500000,
        moneyRange: [400000, 800000],
        servers: [
            { ip: '10.SEATTLE.HQ.1', name: 'STARBUCKS-GLOBAL-HQ', type: 'main', secure: true },
            { ip: '10.SEATTLE.HQ.100', name: 'BOARDROOM-SECRETS', type: 'recipes', secure: true },
            { ip: '10.SEATTLE.HQ.500', name: 'EXECUTIVE-PAYROLL', type: 'money', secure: true, moneyAmount: 800000 },
            { ip: '10.SEATTLE.HQ.999', name: 'INVESTOR-FUND', type: 'money', secure: true, moneyAmount: 600000 }
        ]
    },
    {
        id: 'billionaire_pack',
        name: 'Billionaire Vault Pack',
        price: 1000000,
        moneyRange: [750000, 1500000],
        servers: [
            { ip: '777.CAYMAN.OFF.1', name: 'OFFSHORE-MAINFRAME', type: 'main', secure: true },
            { ip: '777.CAYMAN.OFF.888', name: 'TRILLION-DOLLAR-RECIPES', type: 'recipes', secure: true },
            { ip: '777.CAYMAN.OFF.999', name: 'CAYMAN-BILLIONS', type: 'money', secure: true, moneyAmount: 1500000 },
            { ip: '777.SWISS.BANK.1', name: 'SWISS-MEGA-VAULT', type: 'money', secure: true, moneyAmount: 1200000 },
            { ip: '777.MONACO.ROYAL.13', name: 'ROYAL-HONEYPOT', type: 'trap', secure: false }
        ]
    },
    {
        id: 'ultimate_pack',
        name: '💀 ULTIMATE HACK PACK 💀',
        price: 2500000,
        moneyRange: [1500000, 3000000],
        servers: [
            { ip: '999.SHADOW.GOV.1', name: 'SHADOW-GOVERNMENT-DB', type: 'main', secure: true },
            { ip: '999.AREA51.UFO.42', name: 'ALIEN-COFFEE-FORMULA', type: 'recipes', secure: true },
            { ip: '999.FEDERAL.RES.777', name: 'FEDERAL-RESERVE-TAP', type: 'money', secure: true, moneyAmount: 3000000 },
            { ip: '999.PENTAGON.MIL.500', name: 'PENTAGON-BLACK-BUDGET', type: 'money', secure: true, moneyAmount: 2500000 },
            { ip: '999.NSA.TRAP.666', name: 'NSA-SUPER-HONEYPOT', type: 'trap', secure: false }
        ]
    }
];

// Available Recipes to Steal
const recipes = [
    { name: "Pumpkin Spice Latte", value: 50000, difficulty: 1 },
    { name: "Caramel Macchiato Secret Blend", value: 35000, difficulty: 1 },
    { name: "Frappuccino Base Formula", value: 75000, difficulty: 2 },
    { name: "Cold Brew Concentrate Recipe", value: 25000, difficulty: 1 },
    { name: "Unicorn Frappuccino (Discontinued)", value: 100000, difficulty: 3 }
];

// NPC Dialogues
const npcDialogues = {
    greeting: [
        "Yo! Ready to hack into Starbucks? I've been waiting for this moment my whole life... well, since this morning after my third latte.",
        "Dude! Let's steal some coffee secrets! I'm so hyped I could drink a venti right now. Oh wait, I already did. Five of them.",
        "Welcome back, fellow caffeine criminal! Let's make Starbucks regret ever charging $7 for a drink!"
    ],
    help: [
        "Okay okay, here's the deal: First 'scan' the network, then 'connect' to their server. After that, 'bypass' security (carefully!), then you can 'download recipes' or 'transfer money'. Use 'hide' to cover tracks. Don't mess up or the FBI shows up. No pressure! 😅",
        "Pro tip: The order matters! Scan → Connect → Bypass → Steal stuff → Hide → Exit. Miss a step and you're toast. Like, actual toast. Burnt toast. FBI toast.",
        "LISTEN UP ROOKIE: scan, connect, bypass, then grab the goods. The FBI monitors for suspicious activity, so don't type random stuff!"
    ],
    scan: [
        "Ooooh nice! I'm seeing their network topology now. They've got more servers than they have reasonable drink sizes. What's a 'trenta' anyway?!",
        "Scanning complete! Found their main server: SBUX-CORP-01. Also found a server called 'SECRET-RECIPES'. Subtle, right? 🙄",
        "Their firewall is called 'CaffeineWall'. I'm not even joking. These people are OBSESSED with coffee puns."
    ],
    connect: [
        "We're in! Well, we're at the door. It's like we walked into the Starbucks lobby but for their computers. Now bypass that security!",
        "Connection established! I can already smell the digital coffee beans. Mmmmm... 0s and 1s never smelled so good.",
        "Connected! Their welcome message says 'Have a grande day!' I hate it. Let's steal everything."
    ],
    bypass: [
        "YOOO YOU DID IT! Security bypassed! I used my special 'espresso-injection' attack. Get it? Like SQL injection but... coffee? I'll stop.",
        "We're past security! Their password was 'Venti123'. These billion-dollar companies, I swear... 🤦",
        "Security DOWN! I feel like a coffee ninja. A caffeine-fueled cyber samurai. Okay I need to calm down."
    ],
    download: [
        "JACKPOT! Recipe acquired! Do you know how much hipsters will pay for this?! We're gonna be RICH! Well, coffee-rich!",
        "Got it! Another recipe in the bag! Soon we'll know more about coffee than the baristas. Which honestly isn't hard, JESSICA.",
        "Downloaded! This recipe has like 47 ingredients. How is this even legal to sell? Whatever, it's ours now!"
    ],
    money: [
        "CHA-CHING! 💰 Money transferred! We're basically Robin Hood but instead of giving to the poor, we're keeping it for... more coffee.",
        "SHOW ME THE MONEY! Wait, that's copyrighted. Uh... SHOW ME THE COFFEE FUNDS!",
        "Transfer complete! This is probably enough to buy like... 3 lattes. Have you SEEN their prices?!"
    ],
    hide: [
        "Covering our tracks! I'm replacing our IP address with Dunkin's corporate office. Let them take the fall! 😈",
        "Tracks hidden! I made it look like the hack came from a Keurig machine. They'll never suspect us.",
        "We're ghosts now! Digital phantoms! Coffee-stealing specters! I really need to sleep..."
    ],
    warning: [
        "Yo, careful! Detection is getting high! Maybe 'hide' soon before the FBI shows up. Those guys do NOT mess around.",
        "DUDE DUDE DUDE, the detection meter is climbing! Starbucks' security team is getting suspicious!",
        "Bro, we're getting hot! And not the good kind of hot like fresh coffee. The BAD kind. Like FBI raid kind!"
    ],
    random: [
        "Did you know Starbucks sells like 4 billion cups of coffee a year? We're barely making a dent here.",
        "I once tried to order a 'tall' and they gave me the small one. I've never recovered from that betrayal.",
        "Fun fact: The Starbucks logo is a mermaid. What do mermaids have to do with coffee? NOTHING. It's madness.",
        "My barista wrote 'Have a nice day' on my cup once. It's been 3 years. Still thinking about her.",
        "Okay real talk, their cake pops are actually fire. Not sorry.",
        "If this works, I'm buying that drink with 27 modifications I saw on TikTok. THE WHOLE THING."
    ],
    error: [
        "Whoa whoa WHOA! What are you typing?! That's not a command! The FBI probably flagged that!",
        "BRO WHAT WAS THAT?! Stick to the plan! Random gibberish = FBI party at your house!",
        "That's... not a thing. Did you try to hack them with vibes? Use actual commands!"
    ],
    arrested: [
        "ABORT ABORT! THE FBI IS HERE! I told you not to mess up! I'm out, good luck, bye forever!",
        "It's been real, friend. I'll visit you in coffee jail. Wait, is that a thing? It should be.",
        "NOOOOO! They got us! Tell my mom I loved her lattes! *connection terminated*"
    ],
    win: [
        "WE DID IT! WE ACTUALLY DID IT! We're the greatest coffee hackers in history! Now let's open our own shop!",
        "LEGENDARY! We're basically the Ocean's Eleven of coffee theft! Except there's two of us! Ocean's Two!",
        "I can't believe it worked! Quick, let's leak the Pumpkin Spice formula to the public! FREEDOM FOR ALL!"
    ]
};

// Dangerous commands that trigger FBI
const dangerousCommands = [
    "hack", "virus", "bomb", "attack", "destroy", "delete all", "rm -rf",
    "format", "crash", "ddos", "ransom", "encrypt", "nuke", "explode",
    "shutdown", "kill", "terminate"
];

// DOM Elements
let terminalOutput, terminalInput, npcDialogue, npcName;
let securityLevel, detectionBar, detectionPercent;
let coffeeCount, moneyCount, recipesCount;

// Initialize DOM elements
function initElements() {
    terminalOutput = document.getElementById('terminal-output');
    terminalInput = document.getElementById('terminal-input');
    npcDialogue = document.getElementById('npc-dialogue');
    npcName = document.getElementById('npc-name');
    securityLevel = document.getElementById('security-level');
    detectionBar = document.getElementById('detection-bar');
    detectionPercent = document.getElementById('detection-percent');
    coffeeCount = document.getElementById('coffee-count');
    moneyCount = document.getElementById('money-count');
    recipesCount = document.getElementById('recipes-count');
}

// Screen Management
function showScreen(screenId) {
    document.querySelectorAll('.screen').forEach(screen => {
        screen.classList.remove('active');
    });
    document.getElementById(screenId).classList.add('active');
    
    if (screenId === 'game-screen') {
        terminalInput.focus();
    }
}

// Random dialogue picker
function getRandomDialogue(category) {
    const dialogues = npcDialogues[category];
    return dialogues[Math.floor(Math.random() * dialogues.length)];
}

// Update NPC dialogue with typing effect
function updateNPC(message, mood = 'normal') {
    gameState.npcMood = mood;
    typeWriter(npcDialogue, message, 30);
}

// Typing effect - with cancellation support
let currentTypingTimeout = null;

function typeWriter(element, text, speed = 50) {
    // Cancel any existing typing animation
    if (currentTypingTimeout) {
        clearTimeout(currentTypingTimeout);
        currentTypingTimeout = null;
    }
    
    element.textContent = '';
    let i = 0;
    
    function type() {
        if (i < text.length) {
            element.textContent += text.charAt(i);
            i++;
            currentTypingTimeout = setTimeout(type, speed);
        } else {
            currentTypingTimeout = null;
        }
    }
    type();
}

// Add message to terminal
function addTerminalMessage(message, className = '') {
    const p = document.createElement('p');
    p.textContent = message;
    if (className) p.className = className;
    terminalOutput.appendChild(p);
    terminalOutput.scrollTop = terminalOutput.scrollHeight;
    
    // Play typing sound
    if (window.AudioManager) {
        AudioManager.playTyping();
    }
}

// Update status panel
function updateStatus() {
    // Security Level
    if (!gameState.isConnected) {
        securityLevel.textContent = "🔒 NOT CONNECTED";
        securityLevel.style.color = "#888";
    } else if (!gameState.securityBypassed) {
        securityLevel.textContent = "🔐 HIGH";
        securityLevel.style.color = "#ff3333";
    } else {
        securityLevel.textContent = "🔓 BYPASSED";
        securityLevel.style.color = "#00ff41";
    }
    
    // Detection Risk
    detectionBar.style.width = gameState.detectionRisk + '%';
    detectionPercent.textContent = gameState.detectionRisk + '%';
    
    if (gameState.detectionRisk > 70) {
        detectionBar.style.background = 'linear-gradient(90deg, #ff3333, #ff0000)';
    } else if (gameState.detectionRisk > 40) {
        detectionBar.style.background = 'linear-gradient(90deg, #ffcc00, #ff6600)';
    } else {
        detectionBar.style.background = 'linear-gradient(90deg, #00ff41, #00aa2a)';
    }
    
    // Stats
    coffeeCount.textContent = gameState.coffeeStolen + ' ☕';
    moneyCount.textContent = '$' + gameState.moneyStolen.toLocaleString();
    recipesCount.textContent = gameState.recipesAcquired.length + '/5';
    
    // Day counter
    const dayCount = document.getElementById('day-count');
    if (dayCount) {
        dayCount.textContent = 'Day ' + gameState.currentDay;
    }
}

// Increase detection risk
function increaseDetection(amount) {
    gameState.detectionRisk = Math.min(100, gameState.detectionRisk + amount);
    updateStatus();
    
    if (gameState.detectionRisk >= 100) {
        triggerArrest("Your detection level hit 100%! The FBI traced your connection!");
    } else if (gameState.detectionRisk > 70) {
        updateNPC(getRandomDialogue('warning'), 'worried');
        addTerminalMessage("[WARNING] Detection risk critical! Consider using 'hide' command!", 'warning-msg');
        if (window.AudioManager) AudioManager.playAlert();
    }
}

// Command Handler
function handleCommand(input) {
    const cmd = input.toLowerCase().trim();
    
    // Add to history
    gameState.commandHistory.push(input);
    gameState.historyIndex = gameState.commandHistory.length;
    
    // Display user input
    addTerminalMessage(`> ${input}`, 'user-input');
    
    // Check for dangerous commands first
    for (const dangerous of dangerousCommands) {
        if (cmd.includes(dangerous)) {
            triggerArrest(`You typed "${dangerous}"! The FBI was monitoring for exactly this kind of thing!`);
            return;
        }
    }
    
    // Process commands
    switch(true) {
        case cmd === 'help':
            commandHelp();
            break;
        case cmd === 'scan':
            commandScan();
            break;
        case cmd === 'connect':
            addTerminalMessage("[ERROR] Usage: connect <ip_address>", 'error-msg');
            addTerminalMessage("[INFO] Run 'scan' first to discover available servers.", 'system-msg');
            break;
        case cmd.startsWith('connect '):
            commandConnect(cmd.substring(8).trim());
            break;
        case cmd === 'unlock':
            addTerminalMessage("[ERROR] Usage: unlock <ip_address>", 'error-msg');
            addTerminalMessage("[INFO] Try to unlock a locked server (50/50 chance).", 'system-msg');
            break;
        case cmd.startsWith('unlock '):
            commandUnlock(cmd.substring(7).trim());
            break;
        case cmd === 'bypass':
            commandBypass();
            break;
        case cmd === 'download' || cmd === 'download recipes':
            commandDownload();
            break;
        case cmd === 'transfer' || cmd === 'transfer money':
            commandTransfer();
            break;
        case cmd === 'hide':
            commandHide();
            break;
        case cmd === 'disconnect':
            commandDisconnect();
            break;
        case cmd === 'exit':
            commandExit();
            break;
        case cmd === 'talk':
            commandTalk();
            break;
        case cmd === 'status':
            commandStatus();
            break;
        case cmd === 'clear':
            terminalOutput.innerHTML = '';
            addTerminalMessage("[SYSTEM] Terminal cleared.", 'system-msg');
            break;
        case cmd === 'coffee':
            addTerminalMessage("☕☕☕ COFFEE COFFEE COFFEE ☕☕☕", 'loot-msg');
            updateNPC("Now you're speaking my language! But focus on the mission!", 'happy');
            break;
        case cmd === 'market' || cmd === 'shop' || cmd === 'blackmarket':
            commandMarket();
            break;
        case cmd === 'sell':
            commandSell();
            break;
        case cmd.startsWith('buy '):
            commandBuy(cmd.substring(4).trim());
            break;
        case cmd === 'buy':
            addTerminalMessage("[ERROR] Usage: buy <pack_number>", 'error-msg');
            addTerminalMessage("[INFO] Type 'market' to see available packs.", 'system-msg');
            break;
        case cmd === 'destroy' || cmd === 'destroy everything':
            commandDestroy();
            break;
        case cmd === 'endday' || cmd === 'end day' || cmd === 'nextday':
            commandEndDay();
            break;
        case cmd === '':
            // Empty command, do nothing
            break;
        default:
            commandUnknown(cmd);
    }
}

// Individual Commands
function commandHelp() {
    addTerminalMessage("[SYSTEM] Available commands:", 'system-msg');
    addTerminalMessage("=== HACKING ===", 'success-msg');
    addTerminalMessage("  scan              - Scan the Starbucks network", 'system-msg');
    addTerminalMessage("  connect <ip>      - Connect to server", 'system-msg');
    addTerminalMessage("  bypass            - Bypass security on connected server", 'system-msg');
    addTerminalMessage("  download          - Download recipes from server", 'system-msg');
    addTerminalMessage("  transfer          - Transfer money from server", 'system-msg');
    addTerminalMessage("  hide              - Cover your tracks", 'system-msg');
    addTerminalMessage("  destroy           - Destroy evidence on servers", 'system-msg');
    addTerminalMessage("  disconnect        - Leave current server", 'system-msg');
    addTerminalMessage("=== BLACK MARKET ===", 'loot-msg');
    addTerminalMessage("  market            - View black market (buy/sell)", 'system-msg');
    addTerminalMessage("  sell              - Sell your stolen recipes", 'system-msg');
    addTerminalMessage("  buy <number>      - Buy a server pack", 'system-msg');
    addTerminalMessage("=== OTHER ===", 'warning-msg');
    addTerminalMessage("  endday            - End day (reset detection, keep loot)", 'system-msg');
    addTerminalMessage("  status            - Check current hack status", 'system-msg');
    addTerminalMessage("  talk              - Chat with your hacker buddy", 'system-msg');
    updateNPC(getRandomDialogue('help'), 'helpful');
}

function commandScan() {
    addTerminalMessage("[SCANNING] Initiating network reconnaissance...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage("[SCAN] Target: Starbucks Global Network", 'success-msg');
        addTerminalMessage("[SCAN] Scanning all known IP ranges...", 'success-msg');
        addTerminalMessage("[SCAN] Vulnerability found: CoffeeBeanOverflow v2.3", 'loot-msg');
        addTerminalMessage("", 'system-msg');
        
        // Discover base servers
        gameState.discoveredServers = [...starbucksServers];
        
        addTerminalMessage("[SCAN] ====== US SERVERS ======", 'success-msg');
        starbucksServers.forEach(server => {
            const securityIcon = server.secure ? '🔒' : '🔓';
            addTerminalMessage(`  ${securityIcon} ${server.ip} - ${server.name}`, 'system-msg');
        });
        
        // Add purchased server packs
        serverPacks.forEach(pack => {
            if (gameState.purchasedServerPacks.includes(pack.id)) {
                addTerminalMessage("", 'system-msg');
                addTerminalMessage(`[SCAN] ====== ${pack.name.toUpperCase()} ======`, 'loot-msg');
                pack.servers.forEach(server => {
                    gameState.discoveredServers.push(server);
                    const securityIcon = server.secure ? '🔒' : '🔓';
                    addTerminalMessage(`  ${securityIcon} ${server.ip} - ${server.name}`, 'system-msg');
                });
            }
        });
        
        addTerminalMessage("", 'system-msg');
        addTerminalMessage(`[INFO] Total servers found: ${gameState.discoveredServers.length}`, 'success-msg');
        addTerminalMessage("[INFO] Use 'connect <ip>' to connect to a server", 'system-msg');
        
        gameState.scanCount++;
        increaseDetection(5);
        updateNPC(getRandomDialogue('scan'), 'excited');
        
        if (window.AudioManager) AudioManager.playSuccess();
    }, 1500);
}

function commandConnect(ipAddress) {
    if (gameState.scanCount === 0) {
        addTerminalMessage("[ERROR] No targets found! Run 'scan' first!", 'error-msg');
        updateNPC("Dude, you gotta scan first! We're not psychics here!", 'annoyed');
        increaseDetection(5);
        return;
    }
    
    if (gameState.isConnected) {
        addTerminalMessage(`[INFO] Already connected to ${gameState.connectedServer.name}`, 'warning-msg');
        addTerminalMessage("[INFO] Use 'disconnect' to leave first.", 'system-msg');
        return;
    }
    
    // Find the server by IP
    const server = gameState.discoveredServers.find(s => s.ip.toLowerCase() === ipAddress.toLowerCase());
    
    if (!server) {
        addTerminalMessage(`[ERROR] Unknown IP address: ${ipAddress}`, 'error-msg');
        addTerminalMessage("[INFO] Run 'scan' to see available servers.", 'system-msg');
        updateNPC("That IP doesn't exist, genius! Check the scan results!", 'annoyed');
        increaseDetection(5);
        return;
    }
    
    // Check if it's the honeypot trap - 50/50 chance of getting caught
    if (server.type === 'trap') {
        addTerminalMessage(`[CONNECTING] Establishing connection to ${ipAddress}...`, 'system-msg');
        setTimeout(() => {
            if (Math.random() < 0.5) {
                // Got caught!
                addTerminalMessage("[!!!] HONEYPOT DETECTED [!!!]", 'error-msg');
                addTerminalMessage("[!!!] IT'S A TRAP! [!!!]", 'error-msg');
                triggerArrest("You connected to a honeypot server! The FBI was waiting for you!");
            } else {
                // Lucky escape! But still connect them
                addTerminalMessage("[WARNING] Suspicious server detected...", 'warning-msg');
                addTerminalMessage("[WARNING] Connection unstable...", 'warning-msg');
                addTerminalMessage("[SUCCESS] Phew! Honeypot avoided! Connected anyway!", 'success-msg');
                
                // Actually connect to the server
                gameState.isConnected = true;
                gameState.connectedServer = server;
                updateStatus();
                
                updateNPC("DUDE THAT WAS A HONEYPOT! You got SO lucky! But hey, we're in now!", 'panicked');
                increaseDetection(25);
                if (window.AudioManager) AudioManager.playAlert();
            }
        }, 1500);
        return;
    }
    
    // Set connection immediately so player can type next command right away
    gameState.isConnected = true;
    gameState.connectedServer = server;
    
    addTerminalMessage(`[CONNECTED] You are now connected to ${server.name}!`, 'success-msg');
    addTerminalMessage("[INFO] Type 'bypass' to hack security, 'download' for recipes, 'transfer' for money.", 'system-msg');
    
    increaseDetection(10);
    updateStatus();
    updateNPC(getRandomDialogue('connect'), 'excited');
    if (window.AudioManager) AudioManager.playSuccess();
}

function commandUnlock(ipAddress) {
    if (gameState.scanCount === 0) {
        addTerminalMessage("[ERROR] No targets found! Run 'scan' first!", 'error-msg');
        updateNPC("Scan the network first! How do you expect to unlock something you can't see?", 'annoyed');
        increaseDetection(5);
        return;
    }
    
    // Find the server by IP
    const server = gameState.discoveredServers.find(s => s.ip.toLowerCase() === ipAddress.toLowerCase());
    
    if (!server) {
        addTerminalMessage(`[ERROR] Unknown IP address: ${ipAddress}`, 'error-msg');
        addTerminalMessage("[INFO] Run 'scan' to see available servers.", 'system-msg');
        updateNPC("That IP doesn't exist! Check your scan results!", 'annoyed');
        increaseDetection(5);
        return;
    }
    
    // Check if already unlocked
    if (server.unlocked) {
        addTerminalMessage(`[INFO] ${server.name} is already unlocked!`, 'warning-msg');
        addTerminalMessage("[INFO] Use 'connect' to connect to it.", 'system-msg');
        return;
    }
    
    addTerminalMessage(`[UNLOCK] Attempting to crack ${server.name}...`, 'system-msg');
    addTerminalMessage("[UNLOCK] Running brute force attack...", 'system-msg');
    addTerminalMessage("[UNLOCK] Trying password combinations...", 'system-msg');
    
    setTimeout(() => {
        if (Math.random() < 0.5) {
            // Success! Server unlocked
            server.unlocked = true;
            addTerminalMessage(`[SUCCESS] ${server.name} UNLOCKED!`, 'loot-msg');
            addTerminalMessage("[INFO] You can now connect to this server.", 'success-msg');
            updateNPC("YOOO WE'RE IN! That was some elite hacking right there! 🔓", 'ecstatic');
            increaseDetection(15);
            if (window.AudioManager) AudioManager.playSuccess();
            if (window.GraphicsManager) GraphicsManager.triggerGlitch();
        } else {
            // Failed! Increased detection
            addTerminalMessage("[FAILED] Unlock attempt FAILED!", 'error-msg');
            addTerminalMessage("[WARNING] Security systems detected intrusion attempt!", 'warning-msg');
            updateNPC("Oof, that didn't work! Their security is tighter than I thought. Try again maybe?", 'worried');
            increaseDetection(20);
            if (window.AudioManager) AudioManager.playError();
        }
    }, 2000);
}

function commandBypass() {
    // Check if connected
    if (!gameState.connectedServer) {
        addTerminalMessage("[ERROR] Not connected to any server!", 'error-msg');
        addTerminalMessage("[INFO] Use 'connect <ip>' to connect first.", 'system-msg');
        return;
    }
    
    const server = gameState.connectedServer;
    
    if (server.bypassed) {
        addTerminalMessage(`[INFO] ${server.name} security already bypassed.`, 'warning-msg');
        return;
    }
    
    addTerminalMessage(`[BYPASS] Targeting ${server.name}...`, 'system-msg');
    addTerminalMessage("[BYPASS] Deploying EspressoInjection.exe...", 'system-msg');
    addTerminalMessage("[BYPASS] Cracking password: V*****3...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage("[SUCCESS] PASSWORD CRACKED: Venti123", 'loot-msg');
        addTerminalMessage(`[SUCCESS] ${server.name} security BYPASSED!`, 'success-msg');
        addTerminalMessage("[INFO] Full access granted. You can now download recipes and transfer funds.", 'success-msg');
        
        server.bypassed = true;
        gameState.securityBypassed = true;
        increaseDetection(15);
        updateStatus();
        updateNPC(getRandomDialogue('bypass'), 'ecstatic');
        
        if (window.AudioManager) AudioManager.playSuccess();
        if (window.GraphicsManager) GraphicsManager.triggerGlitch();
    }, 2500);
}

function commandDownload() {
    if (!gameState.isConnected || !gameState.connectedServer) {
        addTerminalMessage("[ERROR] Not connected to any server!", 'error-msg');
        addTerminalMessage("[INFO] Use 'connect <ip>' to connect first.", 'system-msg');
        return;
    }
    
    const server = gameState.connectedServer;
    
    if (!server.bypassed) {
        addTerminalMessage(`[ERROR] Access denied! Bypass security first!`, 'error-msg');
        addTerminalMessage("[INFO] Type 'bypass' to bypass security.", 'system-msg');
        updateNPC("We need to bypass security before we can steal anything! Come on!", 'frustrated');
        return;
    }
    
    const availableRecipes = recipes.filter(r => !gameState.recipesAcquired.includes(r.name));
    
    if (availableRecipes.length === 0) {
        addTerminalMessage("[INFO] All recipes already downloaded!", 'warning-msg');
        return;
    }
    
    const recipe = availableRecipes[0];
    
    addTerminalMessage(`[DOWNLOAD] Accessing ${server.name}...`, 'system-msg');
    addTerminalMessage(`[DOWNLOAD] Locating ${recipe.name}...`, 'system-msg');
    addTerminalMessage("[DOWNLOAD] Decrypting file...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage(`[SUCCESS] Downloaded: ${recipe.name}`, 'loot-msg');
        addTerminalMessage(`[LOOT] Estimated black market value: $${recipe.value.toLocaleString()}`, 'loot-msg');
        
        gameState.recipesAcquired.push(recipe.name);
        gameState.coffeeStolen += 10;
        increaseDetection(12);
        updateStatus();
        updateNPC(getRandomDialogue('download'), 'thrilled');
        
        if (window.AudioManager) AudioManager.playSuccess();
        
        checkWinCondition();
    }, 2000);
}

function commandTransfer() {
    if (!gameState.isConnected || !gameState.connectedServer) {
        addTerminalMessage("[ERROR] Not connected to any server!", 'error-msg');
        addTerminalMessage("[INFO] Use 'connect <ip>' to connect first.", 'system-msg');
        return;
    }
    
    const server = gameState.connectedServer;
    
    if (!server.bypassed) {
        addTerminalMessage(`[ERROR] Access denied! Bypass security first!`, 'error-msg');
        addTerminalMessage("[INFO] Type 'bypass' to bypass security.", 'system-msg');
        updateNPC("No access = no money! Bypass security!", 'frustrated');
        return;
    }
    
    // Use server's moneyAmount if set, otherwise random amount based on server type
    let amount;
    if (server.moneyAmount) {
        amount = server.moneyAmount + Math.floor(Math.random() * (server.moneyAmount * 0.2));
    } else {
        amount = Math.floor(Math.random() * 50000) + 10000;
    }
    
    addTerminalMessage(`[TRANSFER] Accessing ${server.name} financial systems...`, 'system-msg');
    addTerminalMessage("[TRANSFER] Locating offshore accounts...", 'system-msg');
    addTerminalMessage("[TRANSFER] Initiating fund transfer...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage(`[SUCCESS] 💰 Transferred $${amount.toLocaleString()} to your crypto wallet! 💰`, 'loot-msg');
        addTerminalMessage("[INFO] Transaction disguised as 'Coffee Bean Inventory'", 'success-msg');
        
        gameState.moneyStolen += amount;
        increaseDetection(20);
        updateStatus();
        updateNPC(getRandomDialogue('money'), 'ecstatic');
        
        if (window.AudioManager) AudioManager.playSuccess();
        if (window.GraphicsManager) GraphicsManager.showMoneyRain();
        
        checkWinCondition();
    }, 2500);
}

function commandHide() {
    if (gameState.detectionRisk <= 10) {
        addTerminalMessage("[INFO] Detection risk already minimal.", 'warning-msg');
        return;
    }
    
    addTerminalMessage("[HIDE] Covering tracks...", 'system-msg');
    addTerminalMessage("[HIDE] Spoofing IP address...", 'system-msg');
    addTerminalMessage("[HIDE] Redirecting trail to Dunkin' Donuts HQ...", 'system-msg');
    
    setTimeout(() => {
        const reduction = Math.min(gameState.detectionRisk - 10, 40);
        gameState.detectionRisk -= reduction;
        
        addTerminalMessage(`[SUCCESS] Detection risk reduced by ${reduction}%!`, 'success-msg');
        addTerminalMessage("[INFO] Blame successfully shifted to rival coffee chain.", 'success-msg');
        
        gameState.isHidden = true;
        updateStatus();
        updateNPC(getRandomDialogue('hide'), 'sneaky');
        
        if (window.AudioManager) AudioManager.playSuccess();
    }, 2000);
}

function commandDisconnect() {
    if (!gameState.isConnected) {
        addTerminalMessage("[INFO] Not connected to any server.", 'warning-msg');
        return;
    }
    
    const serverName = gameState.connectedServer ? gameState.connectedServer.name : 'server';
    
    addTerminalMessage("[DISCONNECT] Closing connection...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage(`[SUCCESS] Disconnected from ${serverName}.`, 'success-msg');
        addTerminalMessage("[INFO] You can now connect to a different server.", 'system-msg');
        
        gameState.isConnected = false;
        gameState.securityBypassed = false;
        gameState.connectedServer = null;
        updateStatus();
        
        updateNPC("Alright, we're out! Pick another server to hack into. Maybe try the FINANCE-SERVER? 💰", 'excited');
    }, 1000);
}

function commandExit() {
    if (!gameState.isConnected) {
        addTerminalMessage("[INFO] Not connected to any server.", 'warning-msg');
        return;
    }
    
    addTerminalMessage("[EXIT] Closing connection...", 'system-msg');
    addTerminalMessage("[EXIT] Wiping session data...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage("[SUCCESS] Safely disconnected and exited.", 'success-msg');
        
        gameState.isConnected = false;
        gameState.securityBypassed = false;
        gameState.connectedServer = null;
        updateStatus();
        
        checkWinCondition();
    }, 1500);
}

function commandTalk() {
    updateNPC(getRandomDialogue('random'), 'chatty');
    addTerminalMessage("[NPC] Your buddy shares some wisdom...", 'npc-msg');
}

function commandStatus() {
    addTerminalMessage(`=== DAY ${gameState.currentDay} STATUS ===`, 'system-msg');
    addTerminalMessage(`Connected: ${gameState.isConnected ? gameState.connectedServer?.name : 'NO'}`, 'system-msg');
    addTerminalMessage(`Security Bypassed: ${gameState.securityBypassed ? 'YES' : 'NO'}`, 'system-msg');
    addTerminalMessage(`Detection Risk: ${gameState.detectionRisk}%`, gameState.detectionRisk > 70 ? 'error-msg' : 'system-msg');
    addTerminalMessage(`Recipes Stolen: ${gameState.recipesAcquired.length}`, 'system-msg');
    addTerminalMessage(`Recipes Sold: ${gameState.recipesSold.length}`, 'system-msg');
    addTerminalMessage(`Money: $${gameState.moneyStolen.toLocaleString()}`, 'loot-msg');
    addTerminalMessage(`Server Packs Owned: ${gameState.purchasedServerPacks.length}`, 'system-msg');
}

// BLACK MARKET COMMANDS
function commandMarket() {
    addTerminalMessage("", 'system-msg');
    addTerminalMessage("╔════════════════════════════════════════╗", 'loot-msg');
    addTerminalMessage("║      🏴‍☠️ DARK WEB BLACK MARKET 🏴‍☠️      ║", 'loot-msg');
    addTerminalMessage("╚════════════════════════════════════════╝", 'loot-msg');
    addTerminalMessage("", 'system-msg');
    addTerminalMessage(`Your Cash: $${gameState.moneyStolen.toLocaleString()}`, 'success-msg');
    addTerminalMessage("", 'system-msg');
    
    // Show sellable recipes
    const sellableRecipes = gameState.recipesAcquired.filter(r => !gameState.recipesSold.includes(r));
    if (sellableRecipes.length > 0) {
        addTerminalMessage("=== SELL YOUR LOOT ===", 'warning-msg');
        sellableRecipes.forEach(recipeName => {
            const recipe = recipes.find(r => r.name === recipeName);
            if (recipe) {
                addTerminalMessage(`  ${recipeName} - $${recipe.value.toLocaleString()}`, 'system-msg');
            }
        });
        addTerminalMessage("  Type 'sell' to sell ALL recipes", 'system-msg');
    } else {
        addTerminalMessage("=== NO RECIPES TO SELL ===", 'warning-msg');
        addTerminalMessage("  Hack servers and download recipes first!", 'system-msg');
    }
    
    addTerminalMessage("", 'system-msg');
    addTerminalMessage("=== BUY SERVER INTEL ===", 'success-msg');
    
    serverPacks.forEach((pack, index) => {
        const owned = gameState.purchasedServerPacks.includes(pack.id);
        if (owned) {
            addTerminalMessage(`  [${index + 1}] ${pack.name} - OWNED ✓`, 'success-msg');
        } else {
            const moneyServers = pack.servers.filter(s => s.type === 'money');
            const maxMoney = moneyServers.reduce((max, s) => Math.max(max, s.moneyAmount || 0), 0);
            addTerminalMessage(`  [${index + 1}] ${pack.name} - $${pack.price.toLocaleString()}`, 'system-msg');
            addTerminalMessage(`      ${pack.servers.length} servers | 💰 Up to $${maxMoney.toLocaleString()}/transfer`, 'loot-msg');
        }
    });
    
    addTerminalMessage("", 'system-msg');
    addTerminalMessage(`Type 'buy 1' through 'buy ${serverPacks.length}' to purchase`, 'system-msg');
    addTerminalMessage("", 'system-msg');
    addTerminalMessage(`🎯 GOAL: Reach $5,000,000 to WIN!`, 'success-msg');
    
    updateNPC("The black market, baby! Sell your stolen goods, buy intel on more servers. It's a hacker's paradise! 💀", 'excited');
}

function commandSell() {
    const sellableRecipes = gameState.recipesAcquired.filter(r => !gameState.recipesSold.includes(r));
    
    if (sellableRecipes.length === 0) {
        addTerminalMessage("[ERROR] No recipes to sell!", 'error-msg');
        addTerminalMessage("[INFO] Download recipes from servers first.", 'system-msg');
        updateNPC("You got nothing to sell! Go hack some servers and download recipes!", 'annoyed');
        return;
    }
    
    addTerminalMessage("[MARKET] Contacting black market buyer...", 'system-msg');
    addTerminalMessage("[MARKET] Negotiating prices...", 'system-msg');
    
    setTimeout(() => {
        let totalValue = 0;
        sellableRecipes.forEach(recipeName => {
            const recipe = recipes.find(r => r.name === recipeName);
            if (recipe) {
                totalValue += recipe.value;
                gameState.recipesSold.push(recipeName);
                addTerminalMessage(`[SOLD] ${recipeName} for $${recipe.value.toLocaleString()}`, 'loot-msg');
            }
        });
        
        gameState.moneyStolen += totalValue;
        updateStatus();
        
        addTerminalMessage("", 'system-msg');
        addTerminalMessage(`[SUCCESS] Total earned: $${totalValue.toLocaleString()}! 💰`, 'success-msg');
        addTerminalMessage(`[INFO] New balance: $${gameState.moneyStolen.toLocaleString()}`, 'loot-msg');
        
        updateNPC(`KA-CHING! $${totalValue.toLocaleString()} in your pocket! Now buy some server intel and keep hacking!`, 'ecstatic');
        if (window.AudioManager) AudioManager.playSuccess();
    }, 1500);
}

function commandBuy(packNumber) {
    const index = parseInt(packNumber) - 1;
    
    if (isNaN(index) || index < 0 || index >= serverPacks.length) {
        addTerminalMessage("[ERROR] Invalid pack number!", 'error-msg');
        addTerminalMessage("[INFO] Type 'market' to see available packs.", 'system-msg');
        return;
    }
    
    const pack = serverPacks[index];
    
    if (gameState.purchasedServerPacks.includes(pack.id)) {
        addTerminalMessage(`[INFO] You already own ${pack.name}!`, 'warning-msg');
        return;
    }
    
    if (gameState.moneyStolen < pack.price) {
        addTerminalMessage(`[ERROR] Not enough money! Need $${pack.price.toLocaleString()}`, 'error-msg');
        addTerminalMessage(`[INFO] You have: $${gameState.moneyStolen.toLocaleString()}`, 'system-msg');
        updateNPC("You're broke! Hack more servers or sell some recipes first!", 'annoyed');
        return;
    }
    
    addTerminalMessage(`[MARKET] Purchasing ${pack.name}...`, 'system-msg');
    addTerminalMessage("[MARKET] Receiving encrypted data...", 'system-msg');
    
    setTimeout(() => {
        gameState.moneyStolen -= pack.price;
        gameState.purchasedServerPacks.push(pack.id);
        
        addTerminalMessage(`[SUCCESS] ${pack.name} PURCHASED!`, 'success-msg');
        addTerminalMessage(`[INFO] ${pack.servers.length} new servers unlocked!`, 'loot-msg');
        addTerminalMessage("[INFO] Run 'scan' to see new targets!", 'system-msg');
        
        updateNPC(`Nice purchase! You just got intel on ${pack.servers.length} new Starbucks servers worldwide! Run 'scan' to see them!`, 'excited');
        if (window.AudioManager) AudioManager.playSuccess();
        updateStatus();
    }, 1500);
}

function commandDestroy() {
    if (!gameState.isConnected) {
        addTerminalMessage("[ERROR] Not connected to any server!", 'error-msg');
        addTerminalMessage("[INFO] Connect to a server first.", 'system-msg');
        return;
    }
    
    addTerminalMessage("[DESTROY] Initiating evidence destruction...", 'error-msg');
    addTerminalMessage("[DESTROY] Overwriting log files...", 'system-msg');
    addTerminalMessage("[DESTROY] Corrupting backup databases...", 'system-msg');
    addTerminalMessage("[DESTROY] Deploying CoffeeWorm.exe...", 'system-msg');
    
    setTimeout(() => {
        addTerminalMessage("[SUCCESS] 🔥 ALL EVIDENCE DESTROYED! 🔥", 'loot-msg');
        addTerminalMessage("[INFO] Starbucks IT will have a VERY bad morning!", 'success-msg');
        addTerminalMessage("[INFO] Detection risk reduced to 0%!", 'success-msg');
        
        gameState.detectionRisk = 0;
        gameState.serversDestroyed = true;
        gameState.isConnected = false;
        gameState.connectedServer = null;
        gameState.securityBypassed = false;
        updateStatus();
        
        updateNPC("BOOM! Evidence? What evidence? It's all gone! Starbucks is gonna be SO confused tomorrow! 😈🔥", 'ecstatic');
        if (window.AudioManager) AudioManager.playSuccess();
        if (window.GraphicsManager) GraphicsManager.triggerGlitch();
    }, 2500);
}

function commandEndDay() {
    addTerminalMessage("", 'system-msg');
    addTerminalMessage("╔════════════════════════════════════════╗", 'warning-msg');
    addTerminalMessage(`║         END OF DAY ${gameState.currentDay}                  ║`, 'warning-msg');
    addTerminalMessage("╚════════════════════════════════════════╝", 'warning-msg');
    
    // Disconnect if connected
    if (gameState.isConnected) {
        gameState.isConnected = false;
        gameState.connectedServer = null;
        gameState.securityBypassed = false;
    }
    
    addTerminalMessage("[SYSTEM] Logging off for the night...", 'system-msg');
    addTerminalMessage("[SYSTEM] Covering tracks...", 'system-msg');
    
    setTimeout(() => {
        // Save stats for display
        const dayRecipes = gameState.recipesAcquired.length;
        const dayMoney = gameState.moneyStolen;
        
        // Reset for new day
        gameState.currentDay++;
        gameState.detectionRisk = 10;
        gameState.scanCount = 0;
        gameState.discoveredServers = [];
        gameState.serversDestroyed = false;
        
        // Reset server bypass status
        starbucksServers.forEach(s => s.bypassed = false);
        serverPacks.forEach(pack => pack.servers.forEach(s => s.bypassed = false));
        
        addTerminalMessage("", 'system-msg');
        addTerminalMessage(`[NEW DAY] ☀️ DAY ${gameState.currentDay} BEGINS! ☀️`, 'success-msg');
        addTerminalMessage("[INFO] Detection risk reset to 10%", 'success-msg');
        addTerminalMessage("[INFO] Server security has been restored", 'warning-msg');
        addTerminalMessage(`[INFO] Your money: $${dayMoney.toLocaleString()}`, 'loot-msg');
        addTerminalMessage(`[INFO] Your recipes: ${dayRecipes}`, 'system-msg');
        addTerminalMessage("", 'system-msg');
        addTerminalMessage("[TIP] Type 'scan' to start hacking again!", 'system-msg');
        
        updateStatus();
        updateNPC(`Rise and shine, hacker! Day ${gameState.currentDay}! Fresh start, same mission: drain Starbucks dry! ☕💰`, 'excited');
        if (window.AudioManager) AudioManager.playSuccess();
    }, 2000);
}

function commandUnknown(cmd) {
    gameState.wrongCommands++;
    
    addTerminalMessage(`[ERROR] Unknown command: '${cmd}'`, 'error-msg');
    addTerminalMessage("[SYSTEM] Type 'help' for available commands.", 'system-msg');
    
    updateNPC("Typo? No worries, just try again! Type 'help' if you're stuck.", 'helpful');
    
    if (window.AudioManager) AudioManager.playError();
    
    // Only arrest after many wrong commands (more forgiving)
    if (gameState.wrongCommands >= 15) {
        setTimeout(() => {
            triggerArrest("You made WAY too many errors! The FBI got suspicious!");
        }, 1000);
    }
}

// Win condition check
function checkWinCondition() {
    // Win if: stole $5 million (player can stay connected and keep playing)
    if (gameState.moneyStolen >= 5000000 && gameState.detectionRisk < 100) {
        triggerWin();
    }
}

// Trigger arrest
function triggerArrest(reason) {
    gameState.gamePhase = 'arrested';
    
    addTerminalMessage("[!!!] FBI BREACH DETECTED [!!!]", 'error-msg');
    addTerminalMessage("[!!!] TRACING YOUR LOCATION [!!!]", 'error-msg');
    
    updateNPC(getRandomDialogue('arrested'), 'panicked');
    
    if (window.AudioManager) AudioManager.playAlert();
    if (window.GraphicsManager) GraphicsManager.triggerArrestEffect();
    
    setTimeout(() => {
        document.getElementById('arrest-reason').textContent = reason;
        showScreen('arrest-screen');
    }, 2000);
}

// Trigger win
function triggerWin() {
    gameState.gamePhase = 'won';
    
    addTerminalMessage("[!!!] MISSION COMPLETE [!!!]", 'loot-msg');
    
    updateNPC(getRandomDialogue('win'), 'ecstatic');
    
    if (window.AudioManager) AudioManager.playSuccess();
    if (window.GraphicsManager) GraphicsManager.triggerWinEffect();
    
    setTimeout(() => {
        document.getElementById('win-money').textContent = `Money Stolen: $${gameState.moneyStolen.toLocaleString()}`;
        document.getElementById('win-recipes').textContent = `Recipes Acquired: ${gameState.recipesAcquired.length}/5`;
        document.getElementById('win-coffee').textContent = `Virtual Coffee Earned: ${gameState.coffeeStolen} ☕`;
        showScreen('win-screen');
    }, 2000);
}

// Reset game
function resetGame() {
    gameState.isConnected = false;
    gameState.securityBypassed = false;
    gameState.isHidden = false;
    gameState.detectionRisk = 10;
    gameState.coffeeStolen = 0;
    gameState.moneyStolen = 0;
    gameState.recipesAcquired = [];
    gameState.recipesSold = [];
    gameState.commandHistory = [];
    gameState.historyIndex = -1;
    gameState.gamePhase = 'playing';
    gameState.scanCount = 0;
    gameState.wrongCommands = 0;
    gameState.discoveredServers = [];
    gameState.connectedServer = null;
    gameState.currentDay = 1;
    gameState.purchasedServerPacks = [];
    gameState.serversDestroyed = false;
    
    // Reset all server bypass status
    starbucksServers.forEach(s => s.bypassed = false);
    serverPacks.forEach(pack => pack.servers.forEach(s => s.bypassed = false));
    
    // Reset terminal
    terminalOutput.innerHTML = `
        <p class="system-msg">[SYSTEM] Initializing hack sequence...</p>
        <p class="system-msg">[SYSTEM] Loading coffee-based encryption crackers...</p>
        <p class="system-msg">[SYSTEM] Connecting to totally-not-suspicious VPN...</p>
        <p class="success-msg">[SUCCESS] Ready to infiltrate! Type 'help' for commands.</p>
    `;
    
    updateStatus();
    updateNPC(getRandomDialogue('greeting'), 'excited');
}

// Event Listeners
function initEventListeners() {
    // Menu buttons
    document.getElementById('start-hack-btn').addEventListener('click', () => {
        resetGame();
        showScreen('game-screen');
        if (window.AudioManager) AudioManager.startBGMusic();
    });
    
    document.getElementById('instructions-btn').addEventListener('click', () => {
        showScreen('instructions-screen');
    });
    
    document.getElementById('credits-btn').addEventListener('click', () => {
        showScreen('credits-screen');
    });
    
    document.getElementById('exit-game-btn').addEventListener('click', () => {
        window.close();
        // Fallback if window.close() doesn't work (most browsers block it)
        document.body.innerHTML = '<div style="display:flex;justify-content:center;align-items:center;height:100vh;background:#0a0a0a;color:#00ff41;font-family:monospace;font-size:2rem;text-align:center;">Thanks for playing!<br>You can close this tab now. ☕</div>';
    });
    
    document.getElementById('back-to-menu-btn').addEventListener('click', () => {
        showScreen('main-menu');
    });
    
    document.getElementById('back-to-menu-btn-2').addEventListener('click', () => {
        showScreen('main-menu');
    });
    
    document.getElementById('menu-return-btn').addEventListener('click', () => {
        showScreen('main-menu');
        if (window.AudioManager) AudioManager.stopBGMusic();
    });
    
    // Action buttons in status panel
    document.getElementById('market-btn').addEventListener('click', () => {
        if (gameState.gamePhase === 'playing') {
            commandMarket();
        }
    });
    
    document.getElementById('endday-btn').addEventListener('click', () => {
        if (gameState.gamePhase === 'playing') {
            commandEndDay();
        }
    });
    
    // Arrest screen buttons
    document.getElementById('play-again-btn').addEventListener('click', () => {
        resetGame();
        showScreen('game-screen');
    });
    
    document.getElementById('arrest-menu-btn').addEventListener('click', () => {
        showScreen('main-menu');
        if (window.AudioManager) AudioManager.stopBGMusic();
    });
    
    // Win screen buttons
    document.getElementById('play-again-btn-win').addEventListener('click', () => {
        resetGame();
        showScreen('game-screen');
    });
    
    document.getElementById('win-menu-btn').addEventListener('click', () => {
        showScreen('main-menu');
        if (window.AudioManager) AudioManager.stopBGMusic();
    });
    
    // Terminal input
    terminalInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter') {
            const input = terminalInput.value;
            terminalInput.value = '';
            handleCommand(input);
        } else if (e.key === 'ArrowUp') {
            // Command history navigation
            if (gameState.historyIndex > 0) {
                gameState.historyIndex--;
                terminalInput.value = gameState.commandHistory[gameState.historyIndex];
            }
        } else if (e.key === 'ArrowDown') {
            if (gameState.historyIndex < gameState.commandHistory.length - 1) {
                gameState.historyIndex++;
                terminalInput.value = gameState.commandHistory[gameState.historyIndex];
            } else {
                gameState.historyIndex = gameState.commandHistory.length;
                terminalInput.value = '';
            }
        }
    });
    
    // Keep focus on terminal input when game is active
    document.getElementById('game-screen').addEventListener('click', () => {
        terminalInput.focus();
    });
}

// Initialize game
document.addEventListener('DOMContentLoaded', () => {
    initElements();
    initEventListeners();
    updateStatus();
    
    // Start with menu
    showScreen('main-menu');
    
    console.log("☕ STARBUCKS HACKER SIMULATOR LOADED ☕");
    console.log("Remember: This is just a game. Don't actually hack anyone!");
});
